$ErrorActionPreference = "SilentlyContinue"

$BaseDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$LogDir  = Join-Path $BaseDir "logs"
$CfgPath = Join-Path $BaseDir "config.json"
$Starter = Join-Path $BaseDir "start_sender.ps1"

if (!(Test-Path $LogDir)) { New-Item -ItemType Directory -Path $LogDir | Out-Null }
$LogFile = Join-Path $LogDir "sender_poller.log"

function Log($msg) {
  $ts = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
  "$ts $msg" | Out-File -FilePath $LogFile -Append -Encoding UTF8
}

function Read-Config() {
  if (!(Test-Path $CfgPath)) { return $null }
  try {
    return Get-Content $CfgPath -Raw | ConvertFrom-Json
  } catch {
    Log "[!] Failed to parse config.json: $($_.Exception.Message)"
    return $null
  }
}

function Get-Assignment($cfg) {
  if (-not $cfg) { return $null }
  if (-not $cfg.vmhook_url -or -not $cfg.vmhook_token -or -not $cfg.device_id_b64) { return $null }

  $url = "$($cfg.vmhook_url)?device_id_b64=$($cfg.device_id_b64)"
  try {
    $headers = @{ "X-Resnap-Token" = $cfg.vmhook_token }
    return Invoke-RestMethod -Method GET -Uri $url -Headers $headers -TimeoutSec 10
  } catch {
    return $null
  }
}

function Start-Sender($cfg, $peerId) {
  if (!(Test-Path $Starter)) {
    Log "[!] start_sender.ps1 not found: $Starter"
    return
  }

  $signal = $cfg.signal_url
  if (-not $signal) { $signal = "wss://signal.resnap.tech" }

  $mon = 1
  if ($cfg.monitor_index) { $mon = [int]$cfg.monitor_index }

  $w = 1920; $h = 1080; $fps = 60; $br = 12000
  if ($cfg.width) { $w = [int]$cfg.width }
  if ($cfg.height) { $h = [int]$cfg.height }
  if ($cfg.fps) { $fps = [int]$cfg.fps }
  if ($cfg.bitrate_kbps) { $br = [int]$cfg.bitrate_kbps }

  Log "[+] Starting sender: peer=$peerId signal=$signal ${w}x${h}@${fps} br=${br}kbps monitor=$mon"
  powershell.exe -ExecutionPolicy Bypass -File $Starter -PeerId $peerId -SignalUrl $signal -MonitorIndex $mon -Width $w -Height $h -Fps $fps -BitrateKbps $br | Out-Null
}

Log "[*] sender_poller started. cfg=$CfgPath"

$lastPeer = ""
while ($true) {
  $cfg = Read-Config

  if (-not $cfg -or -not $cfg.vmhook_token -or -not $cfg.device_id_b64) {
    Log "[*] Waiting for config (vmhook_token/device_id_b64)..."
    Start-Sleep -Seconds 3
    continue
  }

  $a = Get-Assignment $cfg

  # Expected response format:
  # { ok: true, assigned: true/false, peer_id: "...", token: "...", expires_at: ... }
  if ($a -and $a.ok -and $a.assigned -and $a.peer_id) {
    if ($a.peer_id -ne $lastPeer) {
      Start-Sender $cfg $a.peer_id
      $lastPeer = $a.peer_id
    }
  }

  Start-Sleep -Seconds 2
}