param(
  # Optional: if you already know these during install, you can pass them.
  [string]$VmHookToken = "",
  [string]$DeviceIdB64 = ""
)

$ErrorActionPreference = "Stop"

# Bundle directory is wherever this script is located
$BundleDir = Split-Path -Parent $MyInvocation.MyCommand.Path

# Install/work dir on VM
$TargetDir = "C:\Resnap\Remote"
$LogDir    = Join-Path $TargetDir "logs"
$CfgPath   = Join-Path $TargetDir "config.json"

function Ensure-Dir($p) {
  if (!(Test-Path $p)) { New-Item -ItemType Directory -Path $p | Out-Null }
}

Ensure-Dir $TargetDir
Ensure-Dir $LogDir
$LogFile = Join-Path $LogDir "install_bundle.log"

function Log($msg) {
  $ts = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
  "$ts $msg" | Out-File -FilePath $LogFile -Append -Encoding UTF8
  Write-Host "$ts $msg"
}

function Run-Silent($exe, $args) {
  if (!(Test-Path $exe)) { throw "Missing file: $exe" }
  Log "[*] Running: $exe $args"
  $p = Start-Process -FilePath $exe -ArgumentList $args -Wait -PassThru
  Log "[*] ExitCode: $($p.ExitCode)"
  return $p.ExitCode
}

Log "[*] === Snap Remote Bundle install started ==="
Log "[*] BundleDir: $BundleDir"
Log "[*] TargetDir: $TargetDir"

# 1) Copy runtime files (sender + scripts) into target dir
$toCopy = @("webrtc-sendrecv.exe", "start_sender.ps1", "sender_poller.ps1")
foreach ($f in $toCopy) {
  $src = Join-Path $BundleDir $f
  $dst = Join-Path $TargetDir $f
  if (!(Test-Path $src)) { throw "Missing in bundle: $f" }
  Copy-Item -Force $src $dst
  Log "[+] Copied $f"
}

# 2) Install VC++ (quiet)
# Your file name in screenshot: VC_redist.x64.exe
$VCRedist = Join-Path $BundleDir "VC_redist.x64.exe"
if (Test-Path $VCRedist) {
  # /install /quiet /norestart is the recommended CLI for VC_redist
  Run-Silent $VCRedist "/install /quiet /norestart" | Out-Null
} else {
  Log "[!] VC_redist.x64.exe not found in bundle (skipping)"
}

# 3) Install GStreamer (quiet)
# Your file name: gstreamer-1.0-msvc-x86_64-1.28.0.exe
$GstInstaller = Get-ChildItem -Path $BundleDir -Filter "gstreamer-1.0-msvc-x86_64-*.exe" -ErrorAction SilentlyContinue | Select-Object -First 1
if ($GstInstaller) {
  # GStreamer exe supports /S (NSIS style). If your build uses MSI, we'll adjust later, but /S works for the official exe.
  Run-Silent $GstInstaller.FullName "/S" | Out-Null
} else {
  Log "[!] GStreamer installer not found (skipping)"
}

# 4) Install Virtual Display Driver (quiet)
# Your file name: Virtual.Display.Driver-v24.12.24-setup-x64.exe
$VDDInstaller = Get-ChildItem -Path $BundleDir -Filter "Virtual.Display.Driver-*-setup-x64.exe" -ErrorAction SilentlyContinue | Select-Object -First 1
if ($VDDInstaller) {
  # Many installers accept /S. If yours requires different switches, tell me and I’ll adjust.
  Run-Silent $VDDInstaller.FullName "/S" | Out-Null
} else {
  Log "[!] Virtual Display Driver installer not found (skipping)"
}

# 5) Create/Update config.json (template)
$cfg = @{
  vmhook_url     = "https://vmhook.resnap.tech/api/webrtc-config"
  vmhook_token   = $VmHookToken
  device_id_b64  = $DeviceIdB64
  signal_url     = "wss://signal.resnap.tech"
  monitor_index  = 1
  width          = 1920
  height         = 1080
  fps            = 60
  bitrate_kbps   = 12000
}

$cfg | ConvertTo-Json -Depth 4 | Out-File -FilePath $CfgPath -Encoding UTF8
Log "[+] Wrote config: $CfgPath"

# 6) Create Scheduled Task to run poller on startup (SYSTEM, highest)
$TaskName = "ResnapSenderPoller"
$Poller = Join-Path $TargetDir "sender_poller.ps1"

try {
  schtasks /Delete /TN $TaskName /F | Out-Null
} catch {}

$cmd = "powershell.exe -ExecutionPolicy Bypass -File `"$Poller`""
schtasks /Create /TN $TaskName /SC ONSTART /RU SYSTEM /RL HIGHEST /TR $cmd | Out-Null
Log "[+] Scheduled task created: $TaskName"

Log "[*] === Install finished ==="
Log "[*] IMPORTANT: After VMHOOK register, write vmhook_token + device_id_b64 into config.json (or rerun install_bundle.ps1 with params)."
Log "[*] You can reboot VM now to start poller automatically."

# Optional: Uncomment if you want install always reboot
# Restart-Computer -Force